<?php

namespace App\Console\Commands;

use App\Models\Trade;
use App\Models\User;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CompleteTradeManually extends Command
{
    protected $signature = 'trade:complete {tradeId} {--result=win}';
    protected $description = 'Manually complete a trade with specified result';

    public function handle()
    {
        $tradeId = $this->argument('tradeId');
        $result = $this->option('result');

        if (!in_array($result, ['win', 'loss'])) {
            $this->error('Result must be either "win" or "loss"');
            return 1;
        }

        $trade = Trade::find($tradeId);

        if (!$trade) {
            $this->error("Trade #{$tradeId} not found");
            return 1;
        }

        if ($trade->status === 'completed') {
            $this->error("Trade #{$tradeId} is already completed");
            return 1;
        }

        try {
            DB::transaction(function () use ($trade, $result) {
                $isWin = ($result === 'win');
                
                // Simulate price movement
                $priceDifference = ($isWin ? 1 : -1) * (rand(10, 80) / 100);
                $currentPrice = $trade->entry_price > 0 
                    ? $trade->entry_price * (1 + ($priceDifference / 100))
                    : 100 * (1 + ($priceDifference / 100)); // Fallback if entry price is 0

                // Calculate profit/loss
                $profitAmount = $isWin
                    ? $trade->amount * ($trade->profit_percentage / 100)
                    : -$trade->amount;

                // Update trade
                $trade->status = 'completed';
                $trade->result = $result;
                $trade->close_price = $currentPrice;
                $trade->profit_amount = $profitAmount;
                $trade->price_difference = $priceDifference;
                $trade->completed_at = now();
                $trade->save();

                // Update user balance
                $user = $trade->user;
                $originalStatus = $trade->getOriginal('status');
                
                if ($originalStatus === 'refunded') {
                    // If trade was refunded, deduct the amount first
                    $user->account_bal -= $trade->amount;
                }

                if ($isWin) {
                    // Return stake + profit
                    $winAmount = $trade->amount + $profitAmount;
                    $user->account_bal += $winAmount;
                }
                // If loss, amount stays deducted

                // Update user stats
                $user->total_trades += 1;
                if ($isWin) {
                    $user->total_wins += 1;
                } else {
                    $user->total_losses += 1;
                }
                $user->win_rate = $user->total_trades > 0 ? 
                    round(($user->total_wins / $user->total_trades) * 100, 2) : 0;
                $user->save();

                Log::info('Trade manually completed', [
                    'trade_id' => $trade->id,
                    'user_id' => $user->id,
                    'result' => $result,
                    'profit_amount' => $profitAmount,
                    'new_balance' => $user->account_bal
                ]);
            });

            $this->info("✓ Trade #{$trade->id} completed successfully as {$result}");
            $this->info("Profit/Loss: $" . number_format($trade->profit_amount, 2));
            $this->info("User new balance: $" . number_format($trade->user->account_bal, 2));

            return 0;
        } catch (\Exception $e) {
            $this->error("Failed to complete trade: {$e->getMessage()}");
            Log::error('Manual trade completion failed', [
                'trade_id' => $tradeId,
                'error' => $e->getMessage()
            ]);
            return 1;
        }
    }
}
