<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AnnualReport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class AnnualReportController extends Controller
{
    public function index()
    {
        $reports = AnnualReport::orderBy('report_year', 'desc')->get();
        return view('admin.annual-reports.index', compact('reports'));
    }

    public function create()
    {
        return view('admin.annual-reports.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'report_year' => 'required|integer|min:1900|max:' . (date('Y') + 1),
            'pdf_file' => 'required|file|mimes:pdf|max:10240', // 10MB max
        ]);

        $path = $request->file('pdf_file')->store('annual-reports', 'public');

        AnnualReport::create([
            'name' => $request->name,
            'file_path' => $path,
            'report_year' => $request->report_year,
            'is_active' => true,
        ]);

        return redirect()->route('admin.annual-reports.index')
            ->with('success', 'Annual report added successfully');
    }

    public function destroy(AnnualReport $annualReport)
    {
        // Delete the file from storage
        if (Storage::disk('public')->exists($annualReport->file_path)) {
            Storage::disk('public')->delete($annualReport->file_path);
        }

        // Delete the database record
        $annualReport->delete();

        return redirect()->route('admin.annual-reports.index')
            ->with('success', 'Annual report deleted successfully');
    }

    public function toggle(AnnualReport $annualReport)
    {
        $annualReport->update([
            'is_active' => !$annualReport->is_active
        ]);

        return redirect()->route('admin.annual-reports.index')
            ->with('success', 'Report status updated successfully');
    }
}