<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Message;
use App\Models\User;
use App\Events\NewMessage;
use App\Events\AdminTyping;
use App\Events\ChatEnded;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class ChatController extends Controller
{
    public function getUsers()
    {
        try {
            return User::where('is_admin', 0)
                ->whereExists(function ($query) {
                    $query->select(DB::raw(1))
                        ->from('messages')
                        ->where(function ($q) {
                            $q->where('from_user', DB::raw('users.id'))
                                ->orWhere('to_user', DB::raw('users.id'));
                        });
                })
                ->select('users.id', 'users.name', 'users.is_online', 'users.last_seen')
                ->addSelect([
                    'last_message' => Message::select('message')
                        ->whereRaw('(from_user = users.id OR to_user = users.id)')
                        ->latest()
                        ->limit(1),
                    'unread_count' => Message::selectRaw('COUNT(*)')
                        ->where('from_user', DB::raw('users.id'))
                        ->where('to_user', auth()->id())
                        ->where('read', false)
                ])
                ->orderBy('unread_count', 'desc')
                ->orderBy('last_seen', 'desc')
                ->get();
        } catch (\Exception $e) {
            Log::error('Error in admin getUsers: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to get users'], 500);
        }
    }

    public function getMessages($userId)
    {
        try {
            // Mark messages as read
            Message::where('to_user', auth()->id())
                ->where('from_user', $userId)
                ->where('read', false)
                ->update(['read' => true]);

            // Get messages
            return Message::where(function($query) use ($userId) {
                $query->where('from_user', auth()->id())
                    ->where('to_user', $userId);
            })->orWhere(function($query) use ($userId) {
                $query->where('from_user', $userId)
                    ->where('to_user', auth()->id());
            })
            ->with(['fromUser:id,name,is_admin'])
            ->orderBy('created_at', 'ASC')
            ->get()
            ->map(function ($message) {
                $message->is_admin_message = $message->fromUser->is_admin == 1;
                return $message;
            });
        } catch (\Exception $e) {
            Log::error('Error in admin getMessages: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to get messages'], 500);
        }
    }

    public function sendMessage(Request $request)
    {
        try {
            if (!$request->message && !$request->hasFile('file')) {
                return response()->json(['error' => 'Message or file is required'], 422);
            }

            $message = new Message();
            $message->from_user = auth()->id();
            $message->to_user = $request->user_id;
            $message->message = $request->message ?? '';

            if ($request->hasFile('file')) {
                $file = $request->file('file');
                
                // Validate file
                $request->validate([
                    'file' => 'file|max:10240|mimes:jpeg,png,gif,pdf,doc,docx'
                ]);

                $path = $file->store('chat_files', 'public');
                $message->file_url = Storage::url($path);
                $message->file_name = $file->getClientOriginalName();
                $message->file_type = $file->getMimeType();
            }

            $message->save();
            broadcast(new NewMessage($message))->toOthers();

            return response()->json($message);
        } catch (\Exception $e) {
            Log::error('Error in admin sendMessage: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to send message'], 500);
        }
    }

	public function endChat(Request $request)
{
    try {
        if (!$request->user_id) {
            return response()->json(['error' => 'User ID is required'], 422);
        }

        // Get all messages between admin and this user
        $messages = Message::where(function($query) use ($request) {
            $query->where('from_user', $request->user_id)
                  ->where('to_user', auth()->id());
        })->orWhere(function($query) use ($request) {
            $query->where('from_user', auth()->id())
                  ->where('to_user', $request->user_id);
        })->get();

        // Delete associated files
        foreach ($messages as $message) {
            if ($message->file_url) {
                // Get file path from URL
                $path = str_replace('/storage/', '', $message->file_url);
                if (Storage::disk('public')->exists($path)) {
                    Storage::disk('public')->delete($path);
                }
            }
        }

        // Delete all messages
        Message::where(function($query) use ($request) {
            $query->where('from_user', $request->user_id)
                  ->where('to_user', auth()->id());
        })->orWhere(function($query) use ($request) {
            $query->where('from_user', auth()->id())
                  ->where('to_user', $request->user_id);
        })->delete();

        // Broadcast chat ended event
        broadcast(new ChatEnded($request->user_id))->toOthers();

        return response()->json([
            'success' => true,
            'message' => 'Chat ended and all messages deleted successfully'
        ]);

    } catch (\Exception $e) {
        Log::error('Error ending chat: ' . $e->getMessage());
        return response()->json([
            'success' => false,
            'error' => 'Failed to end chat'
        ], 500);
    }
	}

	public function getUnreadTotal()
	{
    try {
        $count = Message::where('to_user', auth()->id())
            ->where('read', false)
            ->count();

        return response()->json(['count' => $count]);
    } catch (\Exception $e) {
        Log::error('Error getting unread total: ' . $e->getMessage());
        return response()->json(['count' => 0]);
    }
	}

    public function typing(Request $request)
    {
        try {
            if (!$request->user_id) {
                return response()->json(['error' => 'User ID is required'], 422);
            }

            broadcast(new AdminTyping(auth()->id(), $request->user_id))->toOthers();
            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            Log::error('Error in admin typing: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to broadcast typing status'], 500);
        }
    }
}