<?php

namespace App\Livewire\User\Deposit;

use App\Models\PaymentMethod;
use App\Models\Settings;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;
use Livewire\Component;

class StartDeposit extends Component
{
    public string $method;
    public Collection $methods;
    public string $amount;
    public array $cryptoRates = [];
    public bool $loading = true;
    public bool $error = false;

    public function mount(): void
    {
        $this->methods = PaymentMethod::where(function ($query) {
            $query->where('type', 'deposit')
                ->orWhere('type', 'both');
        })->where('status', 'active')->latest()->get();

        // get the last payment method
        if ($this->methods->count() > 0) {
            $this->method = $this->methods->first()->name;
        }

        $this->fetchCryptoRates();
    }

    protected function fetchCryptoRates(): void
    {
        try {
            // Try to get rates from cache first
            $cachedRates = Cache::get('crypto_rates');
            
            if ($cachedRates) {
                $this->cryptoRates = $cachedRates;
                $this->loading = false;
                return;
            }

            // If not in cache, fetch from API
            $response = Http::get('https://api.coingecko.com/api/v3/simple/price', [
                'ids' => 'bitcoin,ethereum,tether,litecoin',
                'vs_currencies' => 'usd'
            ]);

            if ($response->successful()) {
                $data = $response->json();
                
                $this->cryptoRates = [
                    'Bitcoin' => 1 / ($data['bitcoin']['usd'] ?? 1),
                    'Ethereum' => 1 / ($data['ethereum']['usd'] ?? 1),
                    'USDT' => 1 / ($data['tether']['usd'] ?? 1),
                    'Litecoin' => 1 / ($data['litecoin']['usd'] ?? 1)
                ];

                // Cache the rates for 5 minutes
                Cache::put('crypto_rates', $this->cryptoRates, now()->addMinutes(5));
                
                $this->error = false;
            } else {
                throw new \Exception('Failed to fetch rates');
            }
        } catch (\Exception $e) {
            $this->error = true;
            logger()->error('Crypto rates fetch error: ' . $e->getMessage());
        } finally {
            $this->loading = false;
        }
    }

    public function refreshRates()
    {
        $this->loading = true;
        Cache::forget('crypto_rates');
        $this->fetchCryptoRates();
    }

    public function submitPayment()
    {
         sleep(2);
         
        $spaMode = Settings::select('spa_mode')->find(1)->spa_mode;
        
        if (!auth()->user()->can_deposit) {
            session()->flash('message', __('t.not_allowed_to_deposit'));
            return;
        }

        try {
            $method = PaymentMethod::where('name', $this->method)->firstOrFail();
            session()->put('deposit_amount', $this->amount);
            $this->redirect(route('user.deposit.makepayment', ['method' => $method]), $spaMode ? true : false);
        } catch (ModelNotFoundException $e) {
            session()->flash('message', __('t.payment_method_not_found'));
        } catch (\Throwable $th) {
            session()->flash('message', __('t.something_went_wrong'));
        }
    }

    public function render()
    {
        $template = Settings::select('theme')->find(1)->theme;
        return view("{$template}.deposit.start-deposit")
            ->extends("layouts.{$template}")
            ->title('Make a Deposit');
    }
}