<?php

namespace App\Livewire\User\TradeSignal;

use App\Exceptions\SignalErrorException;
use App\Mail\SignalSubscriptionMail;
use App\Models\Settings;
use App\Models\User;
use App\Notifications\User\SignalSubscriptionSuccessful;
use App\Services\SignalService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Livewire\Component;

class Subscribe extends Component
{
    public $duration = 'Choose';
    public $amount;
    public $expire;
    public $telegram_id;
    public $hasSubscribe = false;
    public $inviteLink;
    public $monthly;
    public $yearly;
    public $quarterly;

    /**
     * The mount method is called when the component is initialized.
     */
    public function mount(SignalService $signal)
    {
        try {
            $settings = $signal->settings();
            $this->monthly   = $settings['signal_monthly_fee'];
            $this->yearly    = $settings['signal_yearly_fee'];
            $this->quarterly = $settings['signal_quartly_fee'];
        } catch (SignalErrorException) {
            session()->flash('message', 'Something went wrong, please try again or contact support');
        }
    }

    /**
     * The render method is required by Livewire to return the view.
     */
    public function render()
    {
        $template = Settings::select('theme')->find(1)->theme;
        return view("{$template}.trade-signal.subscribe");
    }

    /**
     * Listen for changes to $duration and set $amount/$expire accordingly.
     */
    public function updatedDuration()
    {
    if ($this->duration === 'Monthly') {
        $this->amount = $this->monthly;
        $this->expire = now()->addDays(30);
    } elseif ($this->duration === 'Quarterly') {
        $this->amount = $this->quarterly;
        $this->expire = now()->addDays(90);
    } elseif ($this->duration === 'Yearly') {
        $this->amount = $this->yearly;
        $this->expire = now()->addDays(365);
    } else {
        $this->amount = null;
        $this->expire = null;
    }
    }


    /**
     * Handle subscription logic via the SignalService.
     */
    public function subscribe(SignalService $signal)
{
    $this->validate([
        'telegram_id' => ['required'],
        'amount'      => ['required', 'numeric', 'gt:0'],
    ]);

    $user = User::find(Auth::user()->id);

    // Check if user has enough balance
    if ($user->account_bal < floatval($this->amount)) {
        session()->flash('message', __('t.signal_insufficient_funds'));
        return;
    }

    try {
        // Attempt to subscribe via the service
        $sub = $signal->subscribe([
            'id'               => $user->id,
            'duration'         => $this->duration,
            'amount'           => $this->amount,
            'expire'           => $this->expire,
            'telegram_user_id' => $this->telegram_id,
        ]);

        // Debit user
        $user->account_bal -= floatval($this->amount);
        $user->save();

        // Subscription details
        $this->inviteLink   = $sub['inviteLink'];
        $this->hasSubscribe = $sub['hasSubscribe'];

        // Check email notification settings
        $settings = Settings::select([
            'id',
            'notifiable_email',
            'receive_signal_subscribe_email',
            'send_signal_subscribe_email'
        ])->find(1);

        if ($settings->send_signal_subscribe_email) {
            // Notify the user (e.g., in-app or email)
            dispatch(function () use ($user) {
                $user->notify(new SignalSubscriptionSuccessful(
                    $this->duration, 
                    $this->amount, 
                    $this->expire, 
                    __('t.signal_subscription_email_user')
                ));
            })->afterResponse();
        }

        if ($settings->receive_signal_subscribe_email) {
            // Notify admin
            dispatch(function () use ($user, $settings) {
                Mail::to($settings->notifiable_email)->send(new SignalSubscriptionMail([
                    'name'       => $user->name,
                    'duration'   => $this->duration,
                    'created_at' => now(),
                ]));
            })->afterResponse();
        }

        session()->flash('success', __('t.signal_subscription_success'));
    } catch (\Throwable $th) {
        session()->flash('message', $th->getMessage());
    }
}


    /**
     * Example placeholder method returning a simple skeleton view if needed.
     */
    public function placeholder(array $params = [])
    {
        return view('livewire.placeholders.skeleton', $params);
    }
}
