<?php

namespace App\Observers;

use App\Models\Trade;
use App\Models\CopyTrade;
use App\Models\CopiedTrade;
use App\Models\User;
use App\Jobs\ProcessTrade;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class TradeObserver
{
    public function created(Trade $trade): void
    {
        // Only process if the trader is being copied
        $copyTrades = CopyTrade::active()
            ->where('trader_id', $trade->user_id)
            ->get();

        foreach ($copyTrades as $copyTrade) {
            try {
                DB::transaction(function () use ($copyTrade, $trade) {
                    // Get follower
                    $follower = User::find($copyTrade->follower_id);

                    // Check if follower has enough balance
                    if ($follower->account_bal < $copyTrade->amount_per_trade) {
                        throw new \Exception('Insufficient balance to copy trade');
                    }

                    // Determine duration - use trader's duration if set to default
                    $duration = $copyTrade->duration === 'default' ? $trade->duration : $copyTrade->duration;

                    // Create copied trade record
                    $copiedTrade = CopiedTrade::create([
                        'copy_trade_id' => $copyTrade->id,
                        'original_trade_id' => $trade->id,
                        'amount' => $copyTrade->amount_per_trade,
                        'status' => 'pending'
                    ]);

                    // Deduct from follower's balance
                    $follower->decrement('account_bal', $copyTrade->amount_per_trade);

                    // Create the actual trade using determined duration
                    $newTrade = Trade::create([
                        'user_id' => $copyTrade->follower_id,
                        'pair' => $trade->pair,
                        'amount' => $copyTrade->amount_per_trade,
                        'type' => $trade->type,
                        'entry_price' => $trade->entry_price,
                        'profit_percentage' => $trade->profit_percentage,
                        'duration' => $duration,
                        'status' => 'active'
                    ]);

                    // Update copied trade with the new trade ID
                    $copiedTrade->update([
                        'copied_trade_id' => $newTrade->id,
                        'status' => 'executed'
                    ]);

                    // Process the copied trade with the determined duration
                    dispatch(new ProcessTrade($newTrade))->delay(
                        $this->getDurationInSeconds($duration)
                    );

                    // Update copy trade stats
                    $copyTrade->updateStats();

                    // Log the successful copy with duration info
                    logger()->info('Trade copied successfully', [
                        'original_trade_id' => $trade->id,
                        'copied_trade_id' => $newTrade->id,
                        'follower_id' => $follower->id,
                        'amount' => $copyTrade->amount_per_trade,
                        'original_duration' => $trade->duration,
                        'applied_duration' => $duration,
                        'is_default_duration' => $copyTrade->duration === 'default'
                    ]);
                });

            } catch (\Exception $e) {
                logger()->error('Failed to copy trade', [
                    'trade_id' => $trade->id,
                    'copy_trade_id' => $copyTrade->id,
                    'error' => $e->getMessage()
                ]);

                // Create failed copy record
                CopiedTrade::create([
                    'copy_trade_id' => $copyTrade->id,
                    'original_trade_id' => $trade->id,
                    'amount' => $copyTrade->amount_per_trade,
                    'status' => 'failed',
                    'error_message' => $e->getMessage()
                ]);

                // Optionally pause copy trading if balance is insufficient
                if (strpos($e->getMessage(), 'Insufficient balance') !== false) {
                    $copyTrade->pause();
                }
            }
        }
    }

    private function getDurationInSeconds($duration)
    {
        return match($duration) {
            '1m' => 60,
            '5m' => 300,
            '15m' => 900,
            '30m' => 1800,
            '1h' => 3600,
            '4h' => 14400,
            '1d' => 86400,
            default => 60,
        };
    }
}