<?php

namespace App\Services;

use App\Models\Signal;
use App\Models\SignalSetting;
use App\Models\SignalSubscription;
use App\Services\TelegramService;
use App\Models\Settings;
use App\Models\User;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Carbon\Carbon;


class SignalService
{
    private TelegramService $telegramService;
    private SignalSetting $signalSettings;
    private Settings $settings;

    public function __construct(TelegramService $telegramService)
    {
        $this->telegramService = $telegramService;
        
        // Retrieve or create initial settings
        $this->settings = Settings::first();
        $this->signalSettings = SignalSetting::firstOrCreate([
            'platform_type' => 'default'
        ], [
            'settings_json' => []
        ]);
    }
    

    public function settings(): array
    {
        if (Cache::has('signal-settings')) {
            return Cache::get('signal-settings');
        }

        // Prepare default settings structure
        $settings = [
            'signal_monthly_fee' => '',
            'signal_quartly_fee' => '',
            'signal_yearly_fee' => '',
            'chat_id' => $this->signalSettings->chat_id ?? '',
            'telegram_bot_api' => $this->settings->telegram_bot_api ?? '',
            'publish_message' => '',
            'result_message' => '',
            'website' => '',
            'telegram_link' => ''
        ];

        // Merge with existing settings from database
        $existingSettings = $this->signalSettings->settings_json ?? [];
        $mergedSettings = array_merge($settings, $existingSettings);

        // Ensure specific keys exist with their expected names
        $mergedSettings['signal_monthly_fee'] = $mergedSettings['monthly'] ?? '';
        $mergedSettings['signal_quartly_fee'] = $mergedSettings['quaterly'] ?? '';
        $mergedSettings['signal_yearly_fee'] = $mergedSettings['yearly'] ?? '';

        Cache::put('signal-settings', $mergedSettings, now()->addHour());

        return $mergedSettings;
    }

    public function updateSignalSettings(array $data): string
    {
        // Update signal settings
        $this->signalSettings->update([
            'chat_id' => $data['telegram_link'] ?? $this->signalSettings->chat_id,
            'settings_json' => $data
        ]);

        // Optionally update Telegram Bot API in main settings
        if (isset($data['telegram_bot_api'])) {
            $this->settings->telegram_bot_api = $data['telegram_bot_api'];
            $this->settings->save();
        }

        // Clear cache
        Cache::forget('signal-settings');

        return 'Signal settings updated successfully';
    }

    public function getChatId(): string
{
    try {
        $availableChats = $this->telegramService->getAvailableChatIds();

        if (empty($availableChats)) {
            throw new \Exception('No chat IDs found');
        }

        // If multiple chats are found, you can choose how to handle it
        // For now, we'll use the first chat ID
        $selectedChat = $availableChats[0];
        $chatId = $selectedChat['id'];

        // Update signal settings with new chat ID
        $this->signalSettings->update([
            'chat_id' => $chatId,
            'settings_json' => array_merge(
                $this->signalSettings->settings_json ?? [],
                ['chat_id' => $chatId]
            )
        ]);

        Cache::forget('signal-settings');

        // Return a message that includes chat details
        return "Chat ID retrieved successfully: {$chatId} (Type: {$selectedChat['type']}, Title: {$selectedChat['title']})";
    } catch (\Exception $e) {
        Log::error('Failed to get Chat ID: ' . $e->getMessage());
        throw new \Exception('Failed to retrieve Chat ID: ' . $e->getMessage());
    }
}

    public function deletChatId(): string
    {
        // Remove chat ID from signal settings
        $this->signalSettings->update([
            'chat_id' => null,
            'settings_json' => array_diff_key(
                $this->signalSettings->settings_json ?? [], 
                ['chat_id' => '']
            )
        ]);

        Cache::forget('signal-settings');

        return 'Chat ID deleted successfully';
    }

    public function signals(string $page): array
    {
        $query = Signal::orderBy('created_at', 'desc')
            ->paginate(15, ['*'], 'page', $page);

        return [
            'data' => $query->items(),
            'next_page_url' => $query->nextPageUrl(),
            'prev_page_url' => $query->previousPageUrl(),
        ];
    }
    
    public function subscribe(array $data): array
    {
    return DB::transaction(function () use ($data) {
        $activeSubscription = SignalSubscription::where('user_id', $data['id'])
            ->where('status', 'active')
            ->first();

        if ($activeSubscription) {
            throw new \Exception('You already have an active subscription.');
        }
        $subscription = SignalSubscription::create([
            'user_id'          => $data['id'],
            'subscribed_at'    => now(),
            'expired_at'       => $this->getSubscriptionExpiry($data['duration'] ?? null),
            'status'           => 'active',
            'telegram_user_id' => $data['telegram_user_id'] ?? null,
            'amount_paid'      => $data['amount'] ?? 0,
            'duration'         => $data['duration'],
        ]);

        $chatIds = $this->telegramService->getAvailableChatIds();
        $inviteLink = ! empty($chatIds)
            ? $this->telegramService->createInviteLink($chatIds[0]['id'])
            : '';
            
        return [
            'id'             => $subscription->id,
            'inviteLink'     => $inviteLink,
            'hasSubscribe'   => true,
            'telegram_user_id' => $data['telegram_user_id'],
        ];
    });
    }
    
    private function getSubscriptionExpiry($duration)
    {
    if ($duration === 'Monthly') {
        return now()->addMonth();
    }

    if ($duration === 'Quarterly') {
        return now()->addMonths(3);
    }

    if ($duration === 'Yearly') {
        return now()->addYear();
    }

    return now()->addDays(is_numeric($duration) ? (int) $duration : 30);
    }


    public function subscription(string $user_id): ?array
    {
        if (Cache::has('signal-subscription-' . $user_id)) {
            return Cache::get('signal-subscription-' . $user_id);
        }

        $subscription = SignalSubscription::where('user_id', $user_id)
            ->where('status', 'active')
            ->with('signal')
            ->latest('expired_at')
            ->first();

        if (!$subscription) {
            return null;
        }

        // Determine subscription type based on duration
        $now = now();
        $expiredAt = $subscription->expired_at;
        $duration = $now->diffInDays($expiredAt) > 300 ? 'Yearly' : 
                    ($now->diffInDays($expiredAt) > 90 ? 'Quarterly' : 'Monthly');

        $subscriptionData = [
            'id' => $subscription->id,
            'user_id' => $subscription->user_id,
            'status' => $subscription->status,
            'amount_paid' => $subscription->amount_paid ?? 0,
            'subscription' => $duration,
            'expired_at' => $subscription->expired_at,
            'reminded_at' => $subscription->reminded_at ?? now()->subDays(7), // Default to a week before expiration
            'signal' => $subscription->signal ? [
                'id' => $subscription->signal->id,
                'pair' => $subscription->signal->pair,
            ] : null
        ];

        Cache::put('signal-subscription-' . $user_id, $subscriptionData, now()->addHour());

        return $subscriptionData;
    }
    
    public function renew(array $data): void
    {
        DB::transaction(function () use ($data) {
        // Example: map duration to a specific number of days
        $daysToAdd = match ($data['duration'] ?? 'Monthly') {
            'Monthly'   => 30,
            'Quarterly' => 90,
            'Yearly'    => 365,
            default     => 30,
        };

        // Find the most recent active subscription
        $subscription = SignalSubscription::where('user_id', $data['id'])
            ->where('status', 'active')
            ->latest('expired_at')
            ->first();

        if ($subscription) {
            // Convert 'expired_at' (string) into a Carbon instance
            $expiredAt = Carbon::parse($subscription->expired_at);

            // If 'expired_at' is in the future, add days to it;
            // otherwise start from now
            $newExpiration = $expiredAt->isFuture()
                ? $expiredAt->addDays($daysToAdd)
                : now()->addDays($daysToAdd);

            // Update subscription
            $subscription->update([
                'expired_at' => $newExpiration,
                'status'     => 'active',
            ]);
        }

        // Clear cached subscription data if applicable
        Cache::forget('signal-subscription-' . $data['id']);
    });
    }


    public function add(array $data): void
    {
        DB::transaction(function () use ($data) {
            $signal = Signal::create([
                'direction' => $data['direction'],
                'pair' => $data['pair'],
                'price' => $data['price'],
                'tp1' => $data['tp1'],
                'tp2' => $data['tp2'],
                'sl1' => $data['sl1'],
                'buy_stop' => $data['buy_stop'],
                'sell_stop' => $data['sell_stop'],
                'buy_limit' => $data['buy_limit'],
                'sell_limit' => $data['sell_limit'],
                'status' => 'unpublished'
            ]);
        });
    }

    public function delete(string $id): void
    {
        DB::transaction(function () use ($id) {
            $signal = Signal::findOrFail($id);
            
            // Send deletion notice to Telegram
            $this->sendSignalDeletionToTelegram($signal);

            $signal->delete();
        });
    }

    public function publish(string $id): array
    {
        return DB::transaction(function () use ($id) {
            $signal = Signal::findOrFail($id);
            $signal->update([
                'status' => 'published',
                'published_at' => now()
            ]);

            // Send publish notification to Telegram
            $this->sendSignalPublishToTelegram($signal);

            // Prepare response for notification
            return [
                'chat_id' => $this->signalSettings->chat_id ?? '',
                'message' => "Signal for {$signal->pair} has been published"
            ];
        });
    }

    public function updateResult(array $data): array
    {
        return DB::transaction(function () use ($data) {
            $signal = Signal::findOrFail($data['signalId']);
            $signal->update([
                'result' => $data['result'],
                'status' => $this->determineSignalStatus($data['result'])
            ]);

            // Send result update to Telegram
            $this->sendSignalResultUpdateToTelegram($signal);

            // Prepare response for notification
            return [
                'chat_id' => $this->signalSettings->chat_id ?? '',
                'message' => "Result for signal {$signal->pair}: {$data['result']}"
            ];
        });
    }
    
    public function subscribers(string $page): array
    {
        if (Cache::has("subscribers-{$page}")) {
            return Cache::get("subscribers-{$page}");
        }

        $query = SignalSubscription::with('user', 'signal')
            ->orderBy('created_at', 'desc')
            ->paginate(15, ['*'], 'page', $page);

        $subscribers = $query->map(function ($subscription) {
            return [
                'id' => $subscription->id,
                'client_id' => $subscription->user_id,
                'subscription' => $subscription->duration,
                'status' => $subscription->status,
                'amount_paid' => $subscription->amount_paid,
                'expired_at' => $subscription->expired_at,
                'created_at' => $subscription->created_at,
            ];
        });
 
        $result = [
            'data' => $subscribers,
            'next_page_url' => $query->nextPageUrl(),
            'prev_page_url' => $query->previousPageUrl(),
        ];

        Cache::put("subscribers-{$page}", $result, now()->addHour());

        return $result;
    }

    public function deleteSubscriber(string $id, string $user_id): string
    {
        return DB::transaction(function () use ($id, $user_id) {
            $subscription = SignalSubscription::where('id', $id)
                ->where('user_id', $user_id)
                ->first();

            if (!$subscription) {
                throw new \Exception('Subscription not found');
            }

            $subscription->delete();

            // Clear cache for all subscriber pages
            for ($i = 1; $i <= 20; $i++) {
                Cache::forget("subscribers-{$i}");
            }

            return 'Subscriber deleted successfully';
        });
    }

   
    public function subscriberBan(string $id, string $banType): string
    {
    return DB::transaction(function () use ($id, $banType) {
        $subscription = SignalSubscription::findOrFail($id);

        $chatId  = $this->signalSettings->chat_id;
        $message = '';

        if ($chatId && $subscription->telegram_user_id) {
            try {
                if ($banType === 'banChatMember') {
                    $subscription->update(['status' => 'banned']);

                    $this->telegramService->banChatMember($chatId, $subscription->telegram_user_id);
                    $message = 'User banned from Telegram group';

                } elseif ($banType === 'unbanChatMember') {
                    $subscription->update(['status' => 'active']);
                    $this->telegramService->unbanChatMember($chatId, $subscription->telegram_user_id);
                    $message = 'User unbanned from Telegram group';

                } else {
                    $message = 'Invalid ban type';
                }
            } catch (\Exception $e) {
                Log::error('Telegram ban/unban failed: ' . $e->getMessage());
                $message = 'Failed to modify Telegram group status';
            }
        } else {
            $message = 'Telegram configuration missing or no Telegram user ID on subscription';
        }

        for ($i = 1; $i <= 20; $i++) {
            Cache::forget("subscribers-{$i}");
        }

        return $message;
    });
    }

    // Telegram Notification Helpers
    private function sendTelegramMessage(string $message): bool
    {
        $chatId = $this->signalSettings->chat_id;
        if (!$chatId) {
            Log::warning('No Telegram Chat ID configured');
            return false;
        }

        return $this->telegramService->sendMessage($chatId, $message);
    }

    private function sendSignalToTelegram(Signal $signal): void
    {
        $message = sprintf(
            "<b>New Trading Signal 📈</b>\n\n" .
            "<b>Pair:</b> %s\n" .
            "<b>Direction:</b> %s\n" .
            "<b>Price:</b> %.2f\n" .
            "<b>Take Profit 1:</b> %.2f\n" .
            "<b>Take Profit 2:</b> %.2f\n" .
            "<b>Stop Loss:</b> %.2f",
            $signal->pair,
            $signal->direction,
            $signal->price,
            $signal->tp1,
            $signal->tp2,
            $signal->sl1
        );

        $this->sendTelegramMessage($message);
    }

    private function sendSignalDeletionToTelegram(Signal $signal): void
    {
        $message = sprintf(
            "<b>Signal Deleted ❌</b>\n\n" .
            "<b>Pair:</b> %s\n" .
            "<b>Deleted At:</b> %s",
            $signal->pair,
            now()->format('Y-m-d H:i:s')
        );

        $this->sendTelegramMessage($message);
    }

    private function sendSignalResultUpdateToTelegram(Signal $signal): void
    {
        $message = sprintf(
            "<b>Signal Result Updated 🔄</b>\n\n" .
            "<b>Pair:</b> %s\n" .
            "<b>Result:</b> %s\n" .
            "<b>Updated At:</b> %s",
            $signal->pair,
            $signal->result,
            now()->format('Y-m-d H:i:s')
        );

        $this->sendTelegramMessage($message);
    }

   private function sendSignalPublishToTelegram(Signal $signal): void
{
    $message = sprintf(
        "<b>Signal Published 🚀</b>\n\n" .
        "<b>Pair:</b> %s\n" .
        "<b>Direction:</b> %s\n" .
        "<b>Entry Price:</b> %.2f\n" .
        "<b>Take Profit 1:</b> %.2f\n" .
        "<b>Take Profit 2:</b> %s\n" .
        "<b>Stop Loss:</b> %.2f\n" .
        "<b>Buy Stop:</b> %s\n" .
        "<b>Sell Stop:</b> %s\n" .
        "<b>Buy Limit:</b> %s\n" .
        "<b>Sell Limit:</b> %s\n" .
        "<b>Published At:</b> %s",
        $signal->pair,
        $signal->direction,
        $signal->price,
        $signal->tp1,
        $signal->tp2 ? number_format($signal->tp2, 2) : 'N/A',
        $signal->sl1,
        $signal->buy_stop ? number_format($signal->buy_stop, 2) : 'N/A',
        $signal->sell_stop ? number_format($signal->sell_stop, 2) : 'N/A',
        $signal->buy_limit ? number_format($signal->buy_limit, 2) : 'N/A',
        $signal->sell_limit ? number_format($signal->sell_limit, 2) : 'N/A',
        now()->format('Y-m-d H:i:s')
    );

    $this->sendTelegramMessage($message);
}

    // Helper to determine signal status based on result
    private function determineSignalStatus(string $result): string
    {
        $result = strtolower($result);
        if (strpos($result, 'win') !== false || strpos($result, 'profit') !== false) {
            return 'won';
        } elseif (strpos($result, 'loss') !== false) {
            return 'lost';
        }
        return 'completed';
    }
}