<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class TelegramService
{
    private ?string $botToken;
    private string $baseUrl;

    public function __construct()
    {
        $settings = \App\Models\Settings::first();
        $this->botToken = $settings->telegram_bot_api ?? null;
        $this->baseUrl = "https://api.telegram.org/bot{$this->botToken}";
    }

    /**
     * Get updates and extract chat IDs
     * 
     * @return array List of found chat IDs with their details
     */
    public function getAvailableChatIds(): array
    {
        if (!$this->botToken) {
            Log::warning('No Telegram Bot Token configured');
            return [];
        }

        try {
            $response = Http::get("{$this->baseUrl}/getUpdates", [
                'limit' => 100,
                'timeout' => 10
            ]);

            if ($response->successful()) {
                $data = $response->json();
                $chatIds = [];

                if (isset($data['result']) && is_array($data['result'])) {
                    foreach ($data['result'] as $update) {
                        // Check my_chat_member updates
                        if (isset($update['my_chat_member']['chat'])) {
                            $chat = $update['my_chat_member']['chat'];
                            $this->addUniqueChatId($chatIds, $chat);
                        }

                        // Check channel_post updates
                        if (isset($update['channel_post']['chat'])) {
                            $chat = $update['channel_post']['chat'];
                            $this->addUniqueChatId($chatIds, $chat);
                        }

                        // Check message updates
                        if (isset($update['message']['chat'])) {
                            $chat = $update['message']['chat'];
                            $this->addUniqueChatId($chatIds, $chat);
                        }
                    }
                }

                return $chatIds;
            }

            Log::error('Failed to get Telegram updates');
            return [];
        } catch (\Exception $e) {
            Log::error('Telegram updates retrieval failed: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Add unique chat ID to the list
     * 
     * @param array &$chatIds Reference to the chat IDs array
     * @param array $chat Chat details
     */
    private function addUniqueChatId(array &$chatIds, array $chat): void
    {
        // Ensure we have an ID and prevent duplicates
        if (!isset($chat['id'])) {
            return;
        }

        $chatId = (string)$chat['id'];
        
        // Check if this chat ID is already in the list
        $exists = false;
        foreach ($chatIds as $existingChat) {
            if ($existingChat['id'] === $chatId) {
                $exists = true;
                break;
            }
        }

        if (!$exists) {
            $chatDetails = [
                'id' => $chatId,
                'type' => $chat['type'] ?? 'unknown',
                'title' => $chat['title'] ?? $chat['username'] ?? $chat['first_name'] ?? 'Unknown',
            ];

            $chatIds[] = $chatDetails;
        }
    }

    /**
     * Send a message via Telegram
     * 
     * @param string $chatId Telegram chat ID
     * @param string $message Message to send
     * @return bool Whether message was sent successfully
     */
    public function sendMessage(string $chatId, string $message): bool
    {
        if (!$this->botToken) {
            Log::warning('No Telegram Bot Token configured');
            return false;
        }

        try {
            $response = Http::post("{$this->baseUrl}/sendMessage", [
                'chat_id' => $chatId,
                'text' => $message,
                'parse_mode' => 'HTML'
            ]);

            return $response->successful();
        } catch (\Exception $e) {
            Log::error('Telegram message send failed: ' . $e->getMessage());
            return false;
        }
    }
    
     public function createInviteLink(string $chatId): string
    {
        if (!$this->botToken) {
            Log::warning('No Telegram Bot Token configured');
            return '';
        }

        try {
            $response = Http::post("{$this->baseUrl}/createChatInviteLink", [
                'chat_id' => $chatId,
                'expire_date' => now()->addDays(30)->timestamp,
                'member_limit' => 1
            ]);

            if ($response->successful()) {
                $result = $response->json();
                return $result['result']['invite_link'] ?? '';
            }

            Log::error('Failed to generate Telegram invite link', [
                'response' => $response->body(),
                'chat_id' => $chatId
            ]);

            return '';
        } catch (\Exception $e) {
            Log::error('Exception generating Telegram invite link: ' . $e->getMessage());
            return '';
        }
    }

    /**
     * Ban a user from a chat
     * 
     * @param string $chatId Telegram chat ID
     * @param string $userId User ID to ban
     * @return bool Whether ban was successful
     */
    public function banChatMember(string $chatId, string $userId): bool
    {
        if (!$this->botToken) {
            Log::warning('No Telegram Bot Token configured');
            return false;
        }

        try {
            $response = Http::post("{$this->baseUrl}/banChatMember", [
                'chat_id' => $chatId,
                'user_id' => $userId
            ]);

            return $response->successful();
        } catch (\Exception $e) {
            Log::error('Telegram ban failed: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Unban a user from a chat
     * 
     * @param string $chatId Telegram chat ID
     * @param string $userId User ID to unban
     * @return bool Whether unban was successful
     */
    public function unbanChatMember(string $chatId, string $userId): bool
    {
        if (!$this->botToken) {
            Log::warning('No Telegram Bot Token configured');
            return false;
        }

        try {
            $response = Http::post("{$this->baseUrl}/unbanChatMember", [
                'chat_id' => $chatId,
                'user_id' => $userId
            ]);

            return $response->successful();
        } catch (\Exception $e) {
            Log::error('Telegram unban failed: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Get bot information
     * 
     * @return array|null Bot information or null if failed
     */
    public function getBotInfo(): ?array
    {
        if (!$this->botToken) {
            Log::warning('No Telegram Bot Token configured');
            return null;
        }

        try {
            $response = Http::get("{$this->baseUrl}/getMe");

            if ($response->successful()) {
                return $response->json()['result'] ?? null;
            }

            Log::error('Failed to get bot information');
            return null;
        } catch (\Exception $e) {
            Log::error('Telegram bot info retrieval failed: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Leave a chat
     * 
     * @param string $chatId Telegram chat ID to leave
     * @return bool Whether leaving the chat was successful
     */
    public function leaveChat(string $chatId): bool
    {
        if (!$this->botToken) {
            Log::warning('No Telegram Bot Token configured');
            return false;
        }

        try {
            $response = Http::post("{$this->baseUrl}/leaveChat", [
                'chat_id' => $chatId
            ]);

            return $response->successful();
        } catch (\Exception $e) {
            Log::error('Telegram leave chat failed: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Get chat information
     * 
     * @param string $chatId Telegram chat ID
     * @return array|null Chat information or null if failed
     */
    public function getChatInfo(string $chatId): ?array
    {
        if (!$this->botToken) {
            Log::warning('No Telegram Bot Token configured');
            return null;
        }

        try {
            $response = Http::get("{$this->baseUrl}/getChat", [
                'chat_id' => $chatId
            ]);

            if ($response->successful()) {
                return $response->json()['result'] ?? null;
            }

            Log::error('Failed to get chat information');
            return null;
        } catch (\Exception $e) {
            Log::error('Telegram chat info retrieval failed: ' . $e->getMessage());
            return null;
        }
    }
}